document.addEventListener('DOMContentLoaded', () => {

    // =======================================================
    // --- HEAD DEVELOPER V4.0 SCRIPT (FINAL & COMPLETE) ---
    // =======================================================

    // --- THEME TOGGLE LOGIC ---
    const themeToggleButton = document.getElementById('theme-toggle');
    const prefersDarkScheme = window.matchMedia('(prefers-color-scheme: dark)');
    const setTheme = (theme) => {
        document.body.setAttribute('data-theme', theme);
        localStorage.setItem('theme', theme);
    };
    const savedTheme = localStorage.getItem('theme');
    if (savedTheme) {
        setTheme(savedTheme);
    } else {
        setTheme(prefersDarkScheme.matches ? 'dark' : 'light');
    }
    themeToggleButton.addEventListener('click', () => {
        const currentTheme = document.body.getAttribute('data-theme');
        const newTheme = currentTheme === 'dark' ? 'light' : 'dark';
        setTheme(newTheme);
    });

    // --- ANIMATION ON SCROLL (INTERSECTION OBSERVER) ---
    try {
        const observer = new IntersectionObserver((entries) => {
            entries.forEach(entry => {
                if (entry.isIntersecting) {
                    const delay = parseInt(entry.target.dataset.delay) || 0;
                    setTimeout(() => entry.target.classList.add('is-visible'), delay);
                    observer.unobserve(entry.target);
                }
            });
        }, { threshold: 0.1 });
        document.querySelectorAll('.animate-on-scroll').forEach(el => observer.observe(el));
    } catch (e) { console.error("Intersection Observer not supported or failed:", e); }

    // --- HERO TYPING ANIMATION ---
    const typingElement = document.getElementById('typing-text');
    if (typingElement) {
        const textToType = "TaklukkanSNBT";
        let index = 0;
        const type = () => {
            if (index < textToType.length) {
                typingElement.textContent += textToType.charAt(index);
                index++;
                setTimeout(type, 150);
            } else {
                setTimeout(() => {
                    typingElement.textContent = "";
                    index = 0;
                    type();
                }, 3000);
            }
        };
        type();
    }

    // --- TESTIMONIALS LOGIC (CLONE & 3D TILT) ---
    const scroller = document.querySelector('.testimonial-scroller');
    if (scroller) {
        const grid = scroller.querySelector('.testimonial-grid');
        const cards = Array.from(grid.children);
        cards.forEach(card => {
            const clone = card.cloneNode(true);
            grid.appendChild(clone);
        });
        document.querySelectorAll('.testimonial-card').forEach(card => {
            card.addEventListener('mousemove', (e) => {
                const rect = card.getBoundingClientRect();
                const x = e.clientX - rect.left;
                const y = e.clientY - rect.top;
                const { width, height } = rect;
                const rotateX = (y - height / 2) / 15;
                const rotateY = (width / 2 - x) / 15;
                card.style.transform = `perspective(1000px) rotateX(${rotateX}deg) rotateY(${rotateY}deg) scale3d(1.05, 1.05, 1.05)`;
            });
            card.addEventListener('mouseleave', () => {
                card.style.transform = 'perspective(1000px) rotateX(0) rotateY(0) scale3d(1, 1, 1)';
            });
        });
    }

    // --- FAQ ACCORDION ---
    document.querySelectorAll('.faq-question').forEach(button => {
        button.addEventListener('click', () => {
            const answer = button.nextElementSibling;
            button.classList.toggle('active');
            answer.style.maxHeight = button.classList.contains('active') ? answer.scrollHeight + 'px' : '0';
        });
    });

    // --- MULTI-STEP FORM LOGIC ---
    const formWrapper = document.getElementById('registrationFormWrapper');
    if (formWrapper) {
        const form = document.getElementById('registrationForm');
        const steps = Array.from(formWrapper.querySelectorAll('.form-step'));
        const nextBtns = formWrapper.querySelectorAll('.next-btn');
        const prevBtns = formWrapper.querySelectorAll('.prev-btn');
        const progressSteps = Array.from(formWrapper.querySelectorAll('.progress-step'));
        let currentStep = 0;

        const showError = (input, message) => {
            const formGroup = input.closest('.form-group');
            const errorDiv = formGroup.querySelector('.error-message');
            input.classList.add('error');
            errorDiv.textContent = message;
        };
        const clearError = (input) => {
            const formGroup = input.closest('.form-group');
            const errorDiv = formGroup.querySelector('.error-message');
            input.classList.remove('error');
            errorDiv.textContent = '';
        };
        const validateStep = (stepIndex) => {
            let isValid = true;
            steps[stepIndex].querySelectorAll('[required]').forEach(input => {
                clearError(input);
                if ((input.type === 'radio' && !form.querySelector(`input[name="${input.name}"]:checked`)) ||
                    (input.type === 'checkbox' && !input.checked) ||
                    (input.type === 'file' && input.files.length === 0) ||
                    (!input.value.trim() && input.type !== 'radio' && input.type !== 'checkbox' && input.type !== 'file')) {
                    showError(input, 'Field ini wajib diisi.');
                    isValid = false;
                } else if (input.id === 'email' && !/^[^\s@]+@[^\s@]+\.[^\s@]+$/.test(input.value)) {
                    showError(input, 'Format email tidak valid.');
                    isValid = false;
                } else if (input.id === 'password' && input.value.length < 6) {
                    showError(input, 'Password minimal 6 karakter.');
                    isValid = false;
                } else if (input.id === 'confirmPassword' && input.value !== document.getElementById('password').value) {
                    showError(input, 'Password tidak cocok.');
                    isValid = false;
                }
            });
            return isValid;
        };
        const showStep = (stepIndex) => {
            steps.forEach((step, index) => step.classList.toggle('active', index === stepIndex));
            progressSteps.forEach((step, index) => step.classList.toggle('active', index <= stepIndex));
        };
        nextBtns.forEach(btn => btn.addEventListener('click', () => {
            if (validateStep(currentStep) && currentStep < steps.length - 1) {
                currentStep++;
                showStep(currentStep);
            }
        }));
        prevBtns.forEach(btn => btn.addEventListener('click', () => {
            if (currentStep > 0) {
                currentStep--;
                showStep(currentStep);
            }
        }));
        form.addEventListener('submit', (e) => {
            e.preventDefault();
            if (validateStep(currentStep)) {
                formWrapper.style.display = 'none';
                document.getElementById('form-success-message').style.display = 'block';
            }
        });
        form.querySelectorAll('input[name="format"]').forEach(radio => radio.addEventListener('change', (e) => {
            document.getElementById('location-wrapper').style.display = e.target.value === 'Offline' ? 'block' : 'none';
        }));
    }

    // --- COUNTDOWN TIMER ---
    const countdownElement = document.getElementById('countdown');
    if (countdownElement) {
        const targetDate = new Date('2025-08-30T23:59:59').getTime();
        const interval = setInterval(() => {
            const now = new Date().getTime();
            const distance = targetDate - now;
            if (distance < 0) {
                clearInterval(interval);
                countdownElement.innerHTML = "Pendaftaran Ditutup";
                return;
            }
            const days = Math.floor(distance / (1000 * 60 * 60 * 24));
            const hours = Math.floor((distance % (1000 * 60 * 60 * 24)) / (1000 * 60 * 60));
            const minutes = Math.floor((distance % (1000 * 60 * 60)) / (1000 * 60));
            countdownElement.innerHTML = `${days} hari ${hours} jam ${minutes} menit`;
        }, 1000);
    }
});